<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class IIF_Data_Syncer {

	public static function init() {
		// Register Cron Event
		if ( ! wp_next_scheduled( 'iif_daily_sync_event' ) ) {
			wp_schedule_event( time(), 'daily', 'iif_daily_sync_event' );
		}
		add_action( 'iif_daily_sync_event', array( __CLASS__, 'run_scheduled_sync' ) );
	}

	public static function run_scheduled_sync() {
		$syncer = new self();
		$syncer->sync_all_products();
	}

	public function sync_all_products() {
		// Check for WooCommerce
		if ( ! class_exists( 'WooCommerce' ) ) {
			return new WP_Error( 'wc_missing', 'WooCommerce is not active.' );
		}

		// 1. Get all published products
		$args = array(
			'status' => 'publish',
			'limit'  => -1,
			'return' => 'ids',
		);
		$products = wc_get_products( $args );

		$api_client = new IIF_API_Client();
		$success_count = 0;
		$errors = array();

		foreach ( $products as $product_id ) {
			$payload = $this->build_payload( $product_id );
			
			// Only send if we have history
			if ( empty( $payload['sales_history'] ) ) {
				continue;
			}

			$response = $api_client->send_forecast_request( $payload );

			if ( is_wp_error( $response ) ) {
				$errors[] = "Product $product_id: " . $response->get_error_message();
			} else {
				$success_count++;
			}
		}

		if ( ! empty( $errors ) ) {
			return new WP_Error( 'sync_partial_fail', "Synced $success_count products. Errors: " . implode( '; ', array_slice( $errors, 0, 5 ) ) );
		}

		return true;
	}

	private function build_payload( $product_id ) {
		$product = wc_get_product( $product_id );

		if ( ! $product ) {
			return array(); // Skip invalid products
		}
		
		// Get Sales History (Last 90 Days) from Orders
		// Note: This is a heavy query. In v2 we should use custom SQL or incremental sync.
		// For MVP, we use wc_get_orders filters.
		
		$date_before = date( 'Y-m-d', strtotime( '+1 day' ) );
		$date_after  = date( 'Y-m-d', strtotime( '-90 days' ) );

		// We need to query orders that contain this product.
		// Standard WC_Order_Query doesn't easily filter by product_id without joins.
		// MVP Approach: Use direct SQL for speed.
		
		global $wpdb;
		
		$sql = "
			SELECT 
				DATE(p.post_date) as order_date, 
				SUM(itm.meta_value) as qty
			FROM {$wpdb->prefix}woocommerce_order_items as oi
			JOIN {$wpdb->prefix}woocommerce_order_itemmeta as itm ON oi.order_item_id = itm.order_item_id
			JOIN {$wpdb->posts} as p ON oi.order_id = p.ID
			WHERE 
				itm.meta_key = '_qty'
				AND oi.order_item_type = 'line_item'
				AND p.post_type = 'shop_order'
				AND p.post_status IN ('wc-completed', 'wc-processing', 'wc-on-hold')
				AND p.post_date >= %s
				AND oi.order_item_id IN (
					SELECT order_item_id FROM {$wpdb->prefix}woocommerce_order_itemmeta 
					WHERE meta_key IN ('_variation_id', '_product_id') 
					AND meta_value = %d
				)
			GROUP BY DATE(p.post_date)
			ORDER BY order_date ASC
		";

		$results = $wpdb->get_results( $wpdb->prepare( $sql, $date_after, $product_id ) );

		$sales_history = array();
		foreach ( $results as $row ) {
			$sales_history[] = array(
				'date'          => $row->order_date,
				'quantity_sold' => (float) $row->qty,
			);
		}

		return array(
			'product_id'       => (string) $product_id, # SaaS uses this for cache/logs
			'sku'              => $product->get_sku(),
			'plan_tier'        => 'starter', // Let backend resolve via API Key, but default here? Actually backend overrides this.
			'current_stock'    => (int) $product->get_stock_quantity(),
			'sales_history'    => $sales_history,
			'forecast_horizon' => 30, // Default
		);
	}
}
